"use client";

import React, { useState } from "react";
import Link from "next/link";
import Modal from "@/components/Helper/Modal";
import {
  useApproveOrderMutation,
  useGetOrderQuery,
} from "@/store/features/order/apiSlice";
import PaginateObserver from "@/components/Helper/PaginateObserver";
import { useDispatch, useSelector } from "react-redux";
import { ordersPageIncrease } from "@/store/features/pagination/paginationSlice";
import ReviewModal from "@/components/Helper/ReviewModal";
import OrderChatBoard from "@/components/Helper/OrderChatBoard";
import StringLang from "@/utilities/StringLang";
import useCurrency from "@/hooks/useCurrency";

function OrderTab() {
  const [toogleModal, setToogleModal] = useState(false);
  const [orderViewType, setOrderViewType] = useState("list");

  const page = useSelector((state) => state.pagination.ordersPage);
  const dispatch = useDispatch();
  const { data, isFetching } = useGetOrderQuery({ page });

  const paginationHandler = () => dispatch(ordersPageIncrease());
  const [approveOrder, { isLoading }] = useApproveOrderMutation();
  const calculatePrice = useCurrency();

  return (
    <div className="w-full md:px-10 md:py-7 p-5 bg-black rounded-md">

      {orderViewType === "list" && (
        <div className="w-full">
          <p className="text-[24px] font-semibold text-white mb-[18px]">
            <StringLang string="My Order" />
          </p>

          <PaginateObserver
            type="onclick"
            handler={paginationHandler}
            currentPage={page}
            lastPage={data?.items?.last_page}
            loading={isFetching}
          >
            <div className="flex flex-col space-y-[18px]">

              {data?.items?.data.map((item) => (
                <Link
                  key={item.id}
                  href={`/product/${item.product_id}`}
                  className="block"
                >
                  <div
                    className="
                      w-full
                      rounded
                      p-2.5
                      border
                      border-[#3C3E42]
                      bg-[#0B0E12]
                      hover:border-primary-blue
                      transition
                      cursor-pointer
                    "
                  >
                    <div className="grid md:grid-cols-2 gap-5 items-center">

                      {/* LEFT */}
                      <div className="flex md:flex-row flex-col gap-4">
                        <div className="md:w-[160px] w-full h-[130px] rounded overflow-hidden">
                          <img
                            src={process.env.BASE_URL + item.variant_image}
                            alt=""
                            className="w-full h-full object-cover"
                          />
                        </div>

                        <div>
                          <span className="text-lg font-semibold text-primary-blue">
                            {calculatePrice(Number(item.option_price))}
                          </span>

                          <h1 className="text-[22px] text-white font-medium">
                            {item.variant_name}
                          </h1>

                          <p className="text-primary-blue text-base">
                            #{item.order_id}
                          </p>
                        </div>
                      </div>

                      {/* RIGHT */}
                      <div className="flex justify-end">
                        <span className="px-6 py-3 bg-primary-blue rounded-lg text-black font-semibold">
                          Produkt ansehen
                        </span>
                      </div>

                    </div>
                  </div>
                </Link>
              ))}

            </div>
          </PaginateObserver>
        </div>
      )}

      {/* ORDER CHAT */}
      {orderViewType !== "list" && orderViewType?.id && (
        <>
          <button onClick={() => setOrderViewType("list")}>⬅</button>

          <OrderChatBoard
            id={orderViewType.id}
            isApprove={orderViewType.approve_by_user === "approved"}
          />
        </>
      )}

      {toogleModal && (
        <Modal onClose={() => setToogleModal(false)}>
          <ReviewModal
            orderId={toogleModal}
            close={() => setToogleModal(false)}
          />
        </Modal>
      )}
    </div>
  );
}

export default OrderTab;
