"use client";

import {
  useGetMessageQuery,
  useSendMessageMutation,
} from "@/store/features/message/apiSlice";
import React, { useEffect, useRef, useState } from "react";
import ReactTimeAgo from "react-time-ago";
import StringLang from "@/utilities/StringLang";
import { useSearchParams } from "next/navigation";

function OrderChatBoard({ id: propId, isApprove }) {
  const searchParams = useSearchParams();
  const orderId = searchParams.get("orderId") || propId;

  const [message, setMessage] = useState("");
  const [image, setImage] = useState(null);

  const { data } = useGetMessageQuery({ id: orderId });
  const [sendMessage, { isLoading }] = useSendMessageMutation();

  const bottomRef = useRef(null);

  /* ⬇️ Auto scroll */
  useEffect(() => {
    bottomRef.current?.scrollIntoView({ behavior: "smooth" });
  }, [data?.messages]);

  /* 🎭 ROLE UI */
  const roleUI = (msg) => {
    const role = Number(msg.send_seller);
    return role === 2
      ? {
          label: "ADMIN",
          align: "justify-start",
          bubble: "bg-blue-600 text-white",
        }
      : {
          label: "YOU",
          align: "justify-end",
          bubble: "bg-green-600 text-white",
        };
  };

  /* 📤 SEND MESSAGE */
  const handleSend = async () => {
    if (!message && !image) return;

    const formData = new FormData();
    formData.append("id", orderId);

    if (message) {
      formData.append("message", message);
    }

    if (image) {
      formData.append("attachment", image);
    }

    await sendMessage(formData);

    setMessage("");
    setImage(null);
  };

  return (
    <div
      className="
        w-full
        h-[75vh] md:h-[70vh]
        flex flex-col
        bg-black
        border border-[#23262B]
        rounded-lg
        overflow-hidden
      "
    >
      {/* ================= CHAT LIST ================= */}
      <div className="flex-1 overflow-y-auto px-3 sm:px-4 py-4 space-y-4">
        {data?.messages?.map((msg) => {
          const ui = roleUI(msg);

          return (
            <div key={msg.id} className={`flex ${ui.align}`}>
              <div className="flex flex-col max-w-[85%] sm:max-w-[65%]">
                {/* META */}
                <span
                  className={`text-[11px] mb-1 ${
                    ui.align === "justify-end"
                      ? "text-right text-gray-400"
                      : "text-left text-gray-400"
                  }`}
                >
                  {ui.label} •{" "}
                  <ReactTimeAgo date={new Date(msg.created_at)} />
                </span>

                {/* MESSAGE BUBBLE */}
                <div
                  className={`
                    px-4 py-3
                    rounded-2xl
                    break-words
                    ${ui.bubble}
                    ${
                      ui.align === "justify-end"
                        ? "rounded-br-md"
                        : "rounded-bl-md"
                    }
                  `}
                >
                  {/* IMAGE */}
                  {msg.attachment && (
                    <img
                      src={`${process.env.NEXT_PUBLIC_ADMIN_URL}/public/uploads/messages/${msg.attachment}`}
                      alt="chat"
                      className="max-w-[220px] rounded-lg mb-2"
                    />
                  )}

                  {/* TEXT (HIDE FALLBACK IMAGE TEXT) */}
                  {msg.message &&
                    msg.message !== "Image Attached" && (
                      <div>{msg.message}</div>
                    )}
                </div>
              </div>
            </div>
          );
        })}

        <div ref={bottomRef} />
      </div>

      {/* ================= INPUT ================= */}
      {!isApprove && (
        <div className="border-t border-[#23262B] bg-black px-3 sm:px-4 py-3">
          <div className="flex flex-col gap-3">
            <textarea
              value={message}
              onChange={(e) => setMessage(e.target.value)}
              placeholder="Type your message…"
              className="
                w-full
                min-h-[70px]
                max-h-[130px]
                resize-none
                rounded-xl
                bg-[#0B0E12]
                border border-primary-blue
                p-3
                text-white
                focus:outline-none
              "
            />

            <input
              type="file"
              accept="image/*"
              onChange={(e) => setImage(e.target.files[0])}
              className="text-white text-sm"
            />

            <button
              onClick={handleSend}
              disabled={isLoading}
              className="
                w-full
                py-3
                rounded-xl
                bg-primary-blue
                text-black
                font-semibold
              "
            >
              <StringLang string={isLoading ? "Sending..." : "Send Message"} />
            </button>
          </div>
        </div>
      )}
    </div>
  );
}

export default OrderChatBoard;
