"use client";
import React, { useEffect } from "react";
import Link from "next/link";
import {
  useAddWishListMutation,
  useDeleteWishListMutation,
  useLazyGetWishListQuery,
} from "@/store/features/wishlist/apiSlice";
import { useRouter } from "next/navigation";
import { useSelector } from "react-redux";
import useCurrency from "@/hooks/useCurrency";
import useLanguage from "@/hooks/useLanguage";

function ProductCard(props) {
  const router = useRouter();
  const isAuth = useSelector((state) => state.auth.accessToken);
  const [getWishListQuery, { data }] = useLazyGetWishListQuery();

  const {
    type = "col",
    thumbnail_image,
    slug,
    regular_price,
    offer_price,
    name,
    total_sale,
    average_rating,
    manual_rating,   // ✅ NOW SUPPORTED
    id,
  } = props;

  const [addWishList, { isLoading }] = useAddWishListMutation();
  const [deleteWishlist] = useDeleteWishListMutation();

  const handleWishlist = (product) => {
    if (!!isAuth) {
      if (data?.products?.find((item) => item?.slug === slug)) {
        deleteWishlist({ product });
      } else {
        addWishList({ product });
      }
    } else {
      router.push("/auth/signin");
    }
  };

  useEffect(() => {
    if (!!isAuth) getWishListQuery();
  }, [isAuth]);

  const calculatePrice = useCurrency();
  const language = useLanguage();

  // ✅ FINAL SAFE RATING (USED EVERYWHERE)
  const finalRating = Number(
    manual_rating !== null && manual_rating !== undefined
      ? manual_rating
      : average_rating ?? 0
  ).toFixed(1);

  /* ========================= ROW VIEW ========================= */
  if (type === "row") {
    return (
      <div
        {...props}
        className={`product-row-item w-full group rounded overflow-hidden ${
          props.className || "bg-[#0B0E12]"
        } border border-[#3C3E42]`}
      >
        <div className="w-full p-2.5 flex rtl:space-x-reverse space-x-4 items-center">
          <div className="w-[160px] h-[126px] relative rounded overflow-hidden">
            <img
              src={process.env.BASE_URL + thumbnail_image}
              alt=""
              className="w-full h-full object-cover transform scale-100 group-hover:scale-105 common-transition"
            />
          </div>

          <div className="flex-1">
            <div className="w-full pr-[14px]">
              {/* Price */}
              <div className="flex rtl:space-x-reverse space-x-2.5 items-center mb-1.5">
                <span className="sm:text-xl text-lg leading-5 font-bold text-primary-blue notranslate">
                  {offer_price
                    ? calculatePrice(Number(offer_price))
                    : calculatePrice(Number(regular_price))}
                </span>
              </div>

              <Link href={`/products/${slug}`}>
                <h1 className="sm:text-xl text-lg text-white lg:leading-[27px] leading-[24px] line-clamp-1 mb-4 font-semibold notranslate">
                  {name}
                </h1>
              </Link>

              <div className="flex justify-between items-center">
                <div className="inline-block py-[6px] px-2.5 rounded-sm bg-[#343434]">
                  <span className="sm:text-sm text-xs font-semibold text-[#A8ABA9]">
                    {total_sale + ` ${language["Sale"]}`}
                  </span>
                </div>

                {/* ✅ FIXED RATING */}
                <div className="flex rtl:space-x-reverse space-x-1 items-center">
                  <span>⭐</span>
                  <span className="sm:text-lg text-base font-medium leading-[21px] text-white">
                    {finalRating}
                  </span>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  /* ========================= COLUMN VIEW ========================= */
  return (
    <div
      {...props}
      className={`product-col-item w-full group rounded overflow-hidden ${
        props.className || "bg-[#0B0E12]"
      } border border-[#3C3E42]`}
    >
      <div className="w-full p-[6px] flex flex-col space-y-4">
        <div className="w-full h-[194px] relative rounded-t overflow-hidden">
          <img
            src={process.env.BASE_URL + thumbnail_image}
            alt=""
            className="w-full h-full object-cover transform scale-100 group-hover:scale-105 common-transition"
          />
        </div>

        <div className="flex-1">
          <div className="w-full px-2 pb-5">
            {/* Price */}
            <div className="flex rtl:space-x-reverse space-x-2.5 items-center mb-1.5">
              <span className="sm:text-xl text-lg leading-6 font-bold text-primary-blue notranslate">
                {offer_price
                  ? calculatePrice(offer_price)
                  : calculatePrice(regular_price)}
              </span>
            </div>

            <Link href={`/products/${slug}`}>
              <h1 className="sm:text-xl text-lg text-white sm:leading-6 font-semibold line-clamp-1 mb-4 notranslate">
                {name}
              </h1>
            </Link>

            <div className="flex justify-between items-center">
              <div className="inline-block py-[6px] px-2.5 rounded-sm bg-[#343434]">
                <span className="sm:text-sm text-xs font-semibold text-[#A8ABA9]">
                  {total_sale + ` ${language["Sale"]}`}
                </span>
              </div>

              {/* ✅ FIXED RATING */}
              <div className="flex rtl:space-x-reverse space-x-1 items-center">
                <span>⭐</span>
                <span className="sm:text-lg text-base font-medium leading-[21px] text-white">
                  {finalRating}
                </span>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}

export default ProductCard;
