"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addLocaleData = addLocaleData;
exports.getDefaultLocale = getDefaultLocale;
exports.getLocaleData = getLocaleData;
exports.resolveLocale = resolveLocale;
exports.setDefaultLocale = setDefaultLocale;
// Fallback locale.
// (when not a single one of the supplied "preferred" locales is available)
var defaultLocale = 'en';

// For all locales added
// their relative time formatter messages will be stored here.
var localesData = {};

// According to the spec BCP 47 language tags are case-insensitive.
// https://tools.ietf.org/html/rfc5646
var lowercaseLocaleLookup = {};
function getDefaultLocale() {
  return defaultLocale;
}
function setDefaultLocale(locale) {
  defaultLocale = locale;
}

/**
 * Gets locale data previously added by `addLocaleData()`.
 * @return  {object} [localeData]
 */
function getLocaleData(locale) {
  return localesData[locale];
}

/**
 * Adds locale data.
 * Is called by `RelativeTimeFormat.addLocale(...)`.
 * @param  {object} localeData
 */
function addLocaleData(localeData) {
  if (!localeData) {
    throw new Error('No locale data passed');
  }
  // This locale data is stored in a global variable
  // and later used when calling `.format(time)`.
  localesData[localeData.locale] = localeData;
  lowercaseLocaleLookup[localeData.locale.toLowerCase()] = localeData.locale;
}

/**
 * Returns a locale for which locale data has been added
 * via `RelativeTimeFormat.addLocale(...)`.
 * @param  {string} locale
 * @return {string} [locale]
 */
function resolveLocale(locale) {
  if (localesData[locale]) {
    return locale;
  }
  if (lowercaseLocaleLookup[locale.toLowerCase()]) {
    return lowercaseLocaleLookup[locale.toLowerCase()];
  }
}
//# sourceMappingURL=LocaleDataStore.js.map