"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = getStep;
var _getStepDenominator = _interopRequireDefault(require("./getStepDenominator.js"));
var _getStepMinTime = _interopRequireDefault(require("./getStepMinTime.js"));
var _round = require("../round.js");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { "default": e }; }
function _typeof(o) { "@babel/helpers - typeof"; return _typeof = "function" == typeof Symbol && "symbol" == typeof Symbol.iterator ? function (o) { return typeof o; } : function (o) { return o && "function" == typeof Symbol && o.constructor === Symbol && o !== Symbol.prototype ? "symbol" : typeof o; }, _typeof(o); }
function ownKeys(e, r) { var t = Object.keys(e); if (Object.getOwnPropertySymbols) { var o = Object.getOwnPropertySymbols(e); r && (o = o.filter(function (r) { return Object.getOwnPropertyDescriptor(e, r).enumerable; })), t.push.apply(t, o); } return t; }
function _objectSpread(e) { for (var r = 1; r < arguments.length; r++) { var t = null != arguments[r] ? arguments[r] : {}; r % 2 ? ownKeys(Object(t), !0).forEach(function (r) { _defineProperty(e, r, t[r]); }) : Object.getOwnPropertyDescriptors ? Object.defineProperties(e, Object.getOwnPropertyDescriptors(t)) : ownKeys(Object(t)).forEach(function (r) { Object.defineProperty(e, r, Object.getOwnPropertyDescriptor(t, r)); }); } return e; }
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == _typeof(i) ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != _typeof(t) || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != _typeof(i)) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); }
/**
 * Finds an appropriate `step` of `steps` for the time interval (in seconds).
 *
 * @param {Object[]} steps - Time formatting steps.
 *
 * @param {number} secondsPassed - Time interval (in seconds).
 *                                 `< 0` for past dates and `> 0` for future dates.
 *
 * @param {number} options.now - Current timestamp.
 *
 * @param {boolean} [options.future] - Whether the date should be formatted as a future one
 *                                     instead of a past one.
 *
 * @param {string} [options.round] - (undocumented) Rounding mechanism.
 *
 * @param {string[]} [options.units] - A list of allowed time units.
 *                                     (Example: ['second', 'minute', 'hour', …])
 *
 * @param {boolean} [options.getNextStep] - Pass true to return `[step, nextStep]` instead of just `step`.
 *
 * @return {Object|Object[]} [step] — Either a `step` or `[prevStep, step, nextStep]`.
 */
function getStep(steps, secondsPassed, _ref) {
  var now = _ref.now,
    future = _ref.future,
    round = _ref.round,
    units = _ref.units,
    getNextStep = _ref.getNextStep;
  // Ignore steps having not-supported time units in `formatAs`.
  steps = filterStepsByUnits(steps, units);
  var step = _getStep(steps, secondsPassed, {
    now: now,
    future: future,
    round: round
  });
  if (getNextStep) {
    if (step) {
      var prevStep = steps[steps.indexOf(step) - 1];
      var nextStep = steps[steps.indexOf(step) + 1];
      return [prevStep, step, nextStep];
    }
    return [undefined, undefined, steps[0]];
  }
  return step;
}
function _getStep(steps, secondsPassed, _ref2) {
  var now = _ref2.now,
    future = _ref2.future,
    round = _ref2.round;
  // If no steps fit the conditions then return nothing.
  if (steps.length === 0) {
    return;
  }

  // Find the most appropriate step.
  var i = getStepIndex(steps, secondsPassed, {
    now: now,
    future: future || secondsPassed < 0,
    round: round
  });

  // If no step is applicable the return nothing.
  if (i === -1) {
    return;
  }
  var step = steps[i];

  // Apply granularity to the time amount
  // (and fall back to the previous step
  //  if the first level of granularity
  //  isn't met by this amount)
  if (step.granularity) {
    // Recalculate the amount of seconds passed based on `granularity`.
    var secondsPassedGranular = (0, _round.getRoundFunction)(round)(Math.abs(secondsPassed) / (0, _getStepDenominator["default"])(step) / step.granularity) * step.granularity;
    // If the granularity for this step is too high,
    // then fall back to the previous step.
    // (if there is any previous step)
    if (secondsPassedGranular === 0 && i > 0) {
      return steps[i - 1];
    }
  }
  return step;
}

/**
 * Iterates through steps until it finds the maximum one satisfying the `minTime` threshold.
 * @param  {Object} steps - Steps.
 * @param  {number} secondsPassed - How much seconds have passed since the date till `now`.
 * @param  {number} options.now - Current timestamp.
 * @param  {boolean} options.future - Whether the time interval should be formatted as a future one.
 * @param  {number} [i] - Gradation step currently being tested.
 * @return {number} Gradation step index.
 */
function getStepIndex(steps, secondsPassed, options) {
  var i = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : 0;
  var minTime = (0, _getStepMinTime["default"])(steps[i], _objectSpread({
    prevStep: steps[i - 1],
    timestamp: options.now - secondsPassed * 1000
  }, options));
  // If `minTime` isn't defined or deduceable for this step, then stop.
  if (minTime === undefined) {
    return i - 1;
  }
  // If the `minTime` threshold for moving from previous step
  // to this step is too high then return the previous step.
  if (Math.abs(secondsPassed) < minTime) {
    return i - 1;
  }
  // If it's the last step then return it.
  if (i === steps.length - 1) {
    return i;
  }
  // Move to the next step.
  return getStepIndex(steps, secondsPassed, options, i + 1);
}

/**
 * Leaves only allowed steps.
 * @param  {Object[]} steps
 * @param  {string[]} units - Allowed time units.
 * @return {Object[]}
 */
function filterStepsByUnits(steps, units) {
  return steps.filter(function (_ref3) {
    var unit = _ref3.unit,
      formatAs = _ref3.formatAs;
    // "unit" is now called "formatAs".
    unit = unit || formatAs;
    // If this step has a `unit` defined
    // then this `unit` must be in the list of allowed `units`.
    if (unit) {
      return units.indexOf(unit) >= 0;
    }
    // A step is not required to specify a `unit`:
    // alternatively, it could specify `format()`.
    // (see "twitter" style for an example)
    return true;
  });
}
//# sourceMappingURL=getStep.js.map