"use client";

import React, { useState } from "react";
import Modal from "@/components/Helper/Modal";
import {
  useApproveOrderMutation,
  useGetOrderQuery,
} from "@/store/features/order/apiSlice";
import PaginateObserver from "@/components/Helper/PaginateObserver";
import { useDispatch, useSelector } from "react-redux";
import { ordersPageIncrease } from "@/store/features/pagination/paginationSlice";
import ReviewModal from "@/components/Helper/ReviewModal";
import OrderChatBoard from "@/components/Helper/OrderChatBoard";
import StringLang from "@/utilities/StringLang";
import useCurrency from "@/hooks/useCurrency";
import Link from "next/link";

function OrderTab() {
  const [toogleModal, setToogleModal] = useState(false);
  const [orderViewType, setOrderViewType] = useState("list");

  const page = useSelector((state) => state?.pagination?.ordersPage);
  const dispatch = useDispatch();

  const { data, isFetching } = useGetOrderQuery({ page });
  const calculatePrice = useCurrency();

  const paginationHandler = () => {
    dispatch(ordersPageIncrease());
  };

  const [approveOrder, { isLoading }] = useApproveOrderMutation();

  return (
    <div className="w-full md:px-10 md:py-7 p-5 bg-black rounded-md">

      {/* ================= ORDER LIST ================= */}
      {orderViewType === "list" && (
        <div className="w-full">
          <p className="text-[24px] font-semibold text-white mb-[18px]">
            <StringLang string="My Order" />
          </p>

          <PaginateObserver
            type="onclick"
            handler={paginationHandler}
            currentPage={page}
            lastPage={data?.items?.last_page}
            loading={isFetching}
          >
            <div className="flex flex-col space-y-[18px]">
              {data?.items?.data.map((item) => (
                <div
                  key={item.id}
                  className="w-full rounded-2xl p-4 border border-white/10 bg-[#0B0E12]/70 backdrop-blur-xl shadow-[0_20px_60px_rgba(0,0,0,0.35)] transition hover:border-white/20 hover:shadow-[0_0_0_1px_rgba(255,255,255,0.08),0_24px_70px_rgba(16,185,129,0.25)]"
                >
                  <div className="grid md:grid-cols-2 gap-5 items-center">

                    {/* LEFT */}
                    <div className="flex gap-4">
                      <div className="w-[140px] h-[120px] rounded-xl overflow-hidden border border-white/10">
                        <img
                          src={process.env.BASE_URL + item.variant_image}
                          alt=""
                          className="w-full h-full object-cover"
                        />
                      </div>

                      <div className="flex flex-col justify-center">
                        <span className="text-lg font-semibold text-emerald-400">
                          {calculatePrice(Number(item.option_price))}
                          {item.qty > 1 && (
                            <span className="ml-2 text-white/80 text-sm">
                              × {item.qty}
                            </span>
                          )}
                        </span>

                        <h1 className="text-[20px] text-white font-semibold mt-1">
                          {item.variant_name}
                        </h1>

                        <p className="text-white/70 text-xs break-all mt-1">
                          Order ID: #{item.track_id}
                        </p>
                      </div>
                    </div>

                    {/* RIGHT */}
                    <div className="flex justify-end">
                      <button
                        onClick={() => setOrderViewType(item)}
                        className="px-6 py-3 rounded-full bg-gradient-to-r from-emerald-400 to-emerald-500 hover:from-white hover:to-white transition hover:shadow-[0_0_0_1px_rgba(255,255,255,0.12),0_12px_30px_rgba(16,185,129,0.45)]"
                      >
                        <span className="text-[#0B0E12] font-semibold">
                          <StringLang string="Get Info" />
                        </span>
                      </button>
                    </div>

                  </div>
                </div>
              ))}
            </div>
          </PaginateObserver>
        </div>
      )}

      {/* ================= ORDER DETAILS ================= */}
      {orderViewType !== "list" && orderViewType?.id && (
        <>
          <div className="flex items-center space-x-2.5 mb-4">
            <button
              onClick={() => setOrderViewType("list")}
              className="text-white text-xl"
            >
              ←
            </button>
            <p className="text-[24px] font-semibold text-white">
              <StringLang string="Order Message" />
            </p>
          </div>

          <div className="w-full flex xl:flex-row flex-col-reverse gap-[30px]">

            {/* CHAT */}
            <div className="flex-1">
              <OrderChatBoard
                id={orderViewType.id}
                isApprove={orderViewType.approve_by_user === "approved"}
              />
            </div>

            {/* ORDER INFO */}
            <div className="xl:w-[280px] w-full xl:border-l border-white/10 pl-4">
              <div className="rounded-2xl border border-white/10 bg-[#0B0E12]/70 backdrop-blur-xl shadow-[0_20px_60px_rgba(0,0,0,0.35)] p-4">
                <p className="text-xs uppercase tracking-[0.2em] text-white/60">
                  <StringLang string="Order Info" />
                </p>
                <p className="text-base font-semibold text-white mt-1 mb-3">
                  <StringLang string="Order Info" />
                </p>

                <div className="w-full h-[170px] relative rounded-xl overflow-hidden border border-white/10 mb-4">
                  <img
                    src={process.env.BASE_URL + orderViewType.variant_image}
                    alt=""
                    className="w-full h-full object-cover"
                  />
                  <div className="absolute inset-0 bg-gradient-to-t from-[#0B0E12]/70 via-transparent to-transparent" />
                </div>

                <span className="text-lg font-semibold text-emerald-400 block">
                  {calculatePrice(Number(orderViewType.option_price))}
                  {orderViewType.qty > 1 && (
                    <span className="ml-2 text-white/80 text-sm">
                      × {orderViewType.qty}
                    </span>
                  )}
                </span>

                <h1 className="text-[20px] text-white mt-1">
                  {orderViewType.option_name}
                </h1>

                <p className="text-white/70 text-xs mt-1 break-all">
                  Order ID: #{orderViewType.track_id}
                </p>

                {/* ACTIONS */}
                <div className="mt-6 flex flex-col gap-3">
                  <div className="h-[46px] flex justify-center items-center rounded-full border border-emerald-400/50 bg-emerald-500/15">
                    <span className="text-emerald-300 font-semibold">
                      Gekauft
                    </span>
                  </div>

                  {orderViewType.approve_by_user !== "approved" && (
                    <button
                      disabled={isLoading}
                      onClick={() =>
                        approveOrder({
                          orderId: orderViewType.id,
                          reSet: () => setOrderViewType("list"),
                        })
                      }
                      className="h-[50px] rounded-full bg-gradient-to-r from-emerald-400 to-emerald-500 hover:from-white hover:to-white transition"
                    >
                      <span className="text-[#0B0E12] font-semibold">
                        Erhalt bestätigen
                      </span>
                    </button>
                  )}

                  <Link
                    href="/auth/profile/tickets"
                    className="h-[46px] rounded-full border border-amber-400/60 text-amber-300 hover:bg-amber-400/10 flex justify-center items-center"
                  >
                    <span className="font-semibold">
                      Support Ticket erstellen
                    </span>
                  </Link>
                </div>
              </div>
            </div>
          </div>
        </>
      )}

      {/* REVIEW MODAL */}
      {toogleModal && (
        <Modal onClose={() => setToogleModal(false)}>
          <ReviewModal
            orderId={toogleModal}
            close={() => setToogleModal(false)}
          />
        </Modal>
      )}
    </div>
  );
}

export default OrderTab;