"use client";

import {
  useGetMessageQuery,
  useSendMessageMutation,
} from "@/store/features/message/apiSlice";
import React, { useEffect, useRef, useState } from "react";
import ReactTimeAgo from "react-time-ago";
import StringLang from "@/utilities/StringLang";
import { useSearchParams } from "next/navigation";

function OrderChatBoard({ id: propId, isApprove }) {
  const searchParams = useSearchParams();
  const orderId = searchParams.get("orderId") || propId;

  const [message, setMessage] = useState(""); // Fixed: Initialize as empty string
  const [image, setImage] = useState(null);

  const { data, isFetching } = useGetMessageQuery(
    { id: orderId },
    { skip: !orderId }
  );

  const [sendMessage, { isLoading }] = useSendMessageMutation();

  const bottomRef = useRef(null);

  /* ================= AUTO SCROLL ================= */
  useEffect(() => {
    if (bottomRef.current) {
      bottomRef.current.scrollIntoView({ behavior: "smooth" });
    }
  }, [data?.messages]);

  /* ================= ROLE UI ================= */
  const roleUI = (msg) => {
    if (Number(msg.send_seller) === 1) {
      return {
        label: "Author",
        align: "justify-start",
        bubble: "bg-white/5 border border-white/10 text-white",
      };
    } else if (Number(msg.send_seller) === 2) {
      return {
        label: "ADMIN",
        align: "justify-start",
        bubble: "bg-blue-500/20 border border-blue-400/30 text-white",
      };
    } else {
      return {
        label: "YOU",
        align: "justify-end",
        bubble: "bg-emerald-500/20 border border-emerald-400/30 text-white",
      };
    }
  };

  /* ================= SEND MESSAGE ================= */
  const handleSend = async () => {
    if (!message && !image) return;

    const formData = new FormData();

    if (message?.trim()) {
      formData.append("message", message.trim());
    }

    if (image) {
      formData.append("attachment", image);
    }

    try {
      await sendMessage({
        formData,
        id: orderId,
        resetForm: () => {
          setMessage("");
          setImage(null);
        },
      }).unwrap();
    } catch (err) {
      console.error("Send failed", err);
    }
  };

  return (
    <div className="w-full h-[calc(100vh-220px)] md:h-[70vh] flex flex-col rounded-2xl border border-white/10 bg-[#0B0E12]/70 backdrop-blur-xl shadow-[0_20px_60px_rgba(0,0,0,0.35)] overflow-hidden">
      <div className="px-4 py-3 border-b border-white/10 flex items-center justify-between">
        <div>
          <p className="text-xs tracking-[0.25em] uppercase text-white/50">
            Order Chat
          </p>
          <p className="text-sm font-semibold text-white">
            <StringLang string="Order Message" />
          </p>
        </div>
        <span
          className={`text-xs font-semibold px-3 py-1 rounded-full border ${
            isApprove
              ? "text-emerald-300 border-emerald-400/30 bg-emerald-500/15"
              : "text-amber-300 border-amber-400/30 bg-amber-500/15"
          }`}
        >
          {isApprove ? "Approved" : "Pending"}
        </span>
      </div>
      {/* ================= CHAT LIST ================= */}
      <div className="flex-1 overflow-y-auto px-4 py-4 space-y-4">
        {data?.messages?.map((msg) => {
          const ui = roleUI(msg);

          return (
            <div key={msg.id} className={`flex ${ui.align}`}>
              <div className="flex flex-col max-w-[75%]">
                {/* META */}
                <span className="text-[11px] text-white/50 mb-1">
                  {ui.label} • <ReactTimeAgo date={new Date(msg.created_at)} />
                </span>

                {/* MESSAGE */}
                <div
                  className={`px-4 py-3 rounded-2xl break-words backdrop-blur-sm ${ui.bubble}`}
                >
                  {/* IMAGE */}
                  {msg.attachment && (
                    <img
                      src={`${process.env.NEXT_PUBLIC_ADMIN_URL}/public/uploads/messages/${msg.attachment}`}
                      alt="chat"
                      className="max-w-[220px] rounded-lg mb-2"
                    />
                  )}

                  {/* TEXT */}
                  {msg.message && msg.message !== "Image Attached" && (
                    <div
                      style={{ whiteSpace: "pre-wrap" }}
                      dangerouslySetInnerHTML={{ __html: msg.message }}
                    />
                  )}
                </div>
              </div>
            </div>
          );
        })}

        <div ref={bottomRef} />
      </div>

      {/* ================= INPUT ================= */}

      <div className="border-t border-white/10 p-3 bg-[#0B0E12]/80">
        <textarea
          value={message}
          onChange={(e) => setMessage(e.target.value)}
          placeholder="Type your message…"
          className="
              w-full
              min-h-[70px]
              max-h-[130px]
              resize-none
              rounded-xl
              bg-white/5
              border border-white/10
              p-3
              text-white
              focus:outline-none
              focus:border-primary-blue/60
              focus:ring-1 focus:ring-primary-blue/40
              mb-2
            "
        />

        <div className="flex items-center gap-3 mb-2">
          <input
            id="chat-attachment"
            type="file"
            accept="image/*"
            onChange={(e) => setImage(e.target.files?.[0] || null)}
            className="sr-only"
          />
          <label
            htmlFor="chat-attachment"
            className="px-4 py-2 rounded-lg border border-white/10 bg-white/5 text-white/80 text-sm font-semibold cursor-pointer hover:bg-white/10 transition"
          >
            <StringLang string="Datei auswählen" />
          </label>
          <span className="text-white/50 text-xs truncate flex-1">
            {image?.name ? image.name : "No file chosen"}
          </span>
        </div>

        <button
          onClick={handleSend}
          disabled={isLoading}
          className="
              w-full
              py-3
              rounded-xl
              bg-gradient-to-r from-primary-blue to-blue-400
              text-black
              font-semibold
              hover:from-white hover:to-white
              disabled:opacity-60
            "
        >
          <StringLang string={isLoading ? "Sending…" : "Send Message"} />
        </button>
      </div>
    </div>
  );
}

export default OrderChatBoard;
