"use client";
import React, { useEffect, useMemo, useState } from "react";
import Link from "next/link";
import {
  useAddWishListMutation,
  useDeleteWishListMutation,
  useLazyGetWishListQuery,
} from "@/store/features/wishlist/apiSlice";
import { useRouter } from "next/navigation";
import { useSelector } from "react-redux";
import useCurrency from "@/hooks/useCurrency";
import useLanguage from "@/hooks/useLanguage";
import { useReviewListQuery } from "@/store/features/reviews/apiSlice";

function shuffleArray(items) {
  const array = [...items];
  for (let i = array.length - 1; i > 0; i -= 1) {
    const j = Math.floor(Math.random() * (i + 1));
    [array[i], array[j]] = [array[j], array[i]];
  }
  return array;
}

function RotatingReview({ reviews, enabled = true }) {
  const list = useMemo(
    () => (enabled ? shuffleArray(reviews) : reviews),
    [reviews, enabled]
  );
  const [index, setIndex] = useState(0);
  const [visible, setVisible] = useState(true);

  useEffect(() => {
    if (!enabled || list.length <= 1) return undefined;

    const interval = setInterval(() => {
      setVisible(false);
      const timeout = setTimeout(() => {
        setIndex((prev) => (prev + 1) % list.length);
        setVisible(true);
      }, 250);

      return () => clearTimeout(timeout);
    }, 4500);

    return () => clearInterval(interval);
  }, [enabled, list.length]);

  if (!list.length) return null;

  return (
    <div className="inline-flex items-start px-3 py-2 rounded-full border border-[#2A2F39] bg-[#10141A]/80 text-[#E2E6EE] text-[11px] font-medium w-full h-full shadow-[inset_0_0_0_1px_rgba(255,255,255,0.02)]">
      <span
        className={`transition-opacity duration-300 break-words whitespace-normal leading-[1.35] line-clamp-3 ${
          visible ? "opacity-100" : "opacity-0"
        }`}
      >
        {list[index]}
      </span>
    </div>
  );
}

function ReviewBox({ reviews, hasReviews, disableReview, rotationEnabled }) {
  if (disableReview) return null;
  if (hasReviews) {
    return rotationEnabled ? (
      <RotatingReview reviews={reviews} enabled />
    ) : (
      <div className="inline-flex items-center px-3 py-2 rounded-full border border-white/15 bg-white/5 backdrop-blur-md text-[#E2E6EE] text-[11px] font-medium w-full h-full shadow-[0_8px_20px_rgba(0,0,0,0.35)]">
        {reviews[0]}
      </div>
    );
  }
  return (
    <div className="inline-flex items-center px-3 py-2 rounded-full border border-white/15 bg-white/5 backdrop-blur-md text-[#8B93A2] text-[11px] font-medium w-full h-full shadow-[0_8px_20px_rgba(0,0,0,0.35)]">
      No reviews yet
    </div>
  );
}

function ProductCard(props) {
  const router = useRouter();
  const isAuth = useSelector((state) => state.auth.accessToken);
  const [getWishListQuery, { data }] = useLazyGetWishListQuery();

  const {
    type = "col",
    thumbnail_image,
    slug,
    regular_price,
    offer_price,
    name,
    total_sale,
    total_sold,
    orders_count,
    sold_count,
    average_rating,
    manual_rating,   // ✅ NOW SUPPORTED
    id,
    product_id,
    parent_id,
    variant_id,
    variation_id,
    reviews,
    disableReview: disableReviewProp,
    has_verified_sale,
  } = props;

  const productId =
    variant_id ??
    variation_id ??
    product_id ??
    id ??
    parent_id ??
    slug; // fallback to slug so variants still fetch reviews

  const saleCount = Number(total_sale ?? total_sold ?? orders_count ?? sold_count ?? 0);
  const rotationEnabled = saleCount > 0;
  const disableReview = disableReviewProp === true; // only explicit flag hides the UI

  const [addWishList, { isLoading }] = useAddWishListMutation();
  const [deleteWishlist] = useDeleteWishListMutation();

  const handleWishlist = (product) => {
    if (!!isAuth) {
      if (data?.products?.find((item) => item?.slug === slug)) {
        deleteWishlist({ product });
      } else {
        addWishList({ product });
      }
    } else {
      router.push("/auth/signin");
    }
  };

  useEffect(() => {
    if (!!isAuth) getWishListQuery();
  }, [isAuth]);

  const calculatePrice = useCurrency();
  const language = useLanguage();

  // ✅ FINAL SAFE RATING (USED EVERYWHERE)
  const finalRating = Number(
    manual_rating !== null && manual_rating !== undefined
      ? manual_rating
      : average_rating ?? 0
  ).toFixed(1);

  const { data: reviewData } = useReviewListQuery(
    { id: productId, page: 1 },
    {
      skip: disableReview || !productId, // ✅ allow zero-sale products to fetch reviews
      refetchOnMountOrArgChange: true,
      refetchOnReconnect: true,
      refetchOnFocus: true,
    }
  );

  const reviewTexts = disableReview
    ? []
    : (Array.isArray(reviews) ? reviews : [])
        .map((r) => r?.comment || r?.review || r?.message || r?.text)
        .filter(Boolean);

  const apiReviewSource =
    reviewData?.reviews?.data ||
    reviewData?.data?.data ||
    reviewData?.data ||
    [];

  const apiReviewTexts = disableReview
    ? []
    : apiReviewSource
        .map((r) => r?.comment || r?.review || r?.message || r?.text)
        .filter(Boolean);

  const finalReviewTexts = reviewTexts.length > 0 ? reviewTexts : apiReviewTexts;
  const hasReviews = !disableReview && finalReviewTexts.length > 0;

  /* ========================= ROW VIEW ========================= */
  if (type === "row") {
    return (
      <Link href={`/products/${slug}`} className="block">
        <div
          {...props}
          className={`product-row-item w-full group rounded-[22px] overflow-hidden relative ${
            props.className || "bg-[#0B0E14]"
          } border border-[#222734] shadow-[0_12px_30px_rgba(0,0,0,0.5)] transition-all duration-300 hover:-translate-y-1 hover:shadow-[0_26px_70px_rgba(0,0,0,0.65)] hover:border-[#364055]`}
        >
          <div className="pointer-events-none absolute inset-0 opacity-0 group-hover:opacity-100 transition-opacity duration-300 bg-[radial-gradient(120%_120%_at_0%_0%,rgba(80,140,255,0.18),transparent_60%)]" />
          <div className="absolute left-0 top-0 h-[3px] w-full bg-gradient-to-r from-[#6AE0FF] via-[#7C5CFF] to-[#00F0A8] opacity-70" />
          <div className="w-full p-4 flex rtl:space-x-reverse space-x-4 items-center">
            <div className="w-[180px] h-[140px] relative rounded-2xl overflow-hidden bg-gradient-to-br from-[#0F141B] to-[#0A0D12] flex items-center justify-center p-3 ring-1 ring-[#1B2230] group-hover:ring-[#2B3550] transition-all duration-300">
              <img
                src={process.env.BASE_URL + thumbnail_image}
                alt=""
                className="w-full h-full object-contain transform scale-100 group-hover:scale-[1.05] transition-all duration-300 max-w-[92%] max-h-[92%] drop-shadow-[0_12px_30px_rgba(0,0,0,0.7)]"
              />
            </div>

            <div className="flex-1">
              <div className="w-full pr-[10px]">
                {/* Price */}
                <div className="flex rtl:space-x-reverse space-x-2 items-center mb-1.5">
                  <span className="sm:text-xl text-lg leading-5 font-bold text-[#7CFFB2] notranslate">
                    {offer_price
                      ? calculatePrice(Number(offer_price))
                      : calculatePrice(Number(regular_price))}
                  </span>
                  {offer_price && (
                    <span className="text-xs text-[#8B93A2] line-through notranslate">
                      {calculatePrice(Number(regular_price))}
                    </span>
                  )}
                </div>

                <h1 className="sm:text-xl text-lg text-white lg:leading-[27px] leading-[24px] line-clamp-3 mb-3 font-semibold notranslate">
                  {name}
                </h1>

                <div className="flex items-center justify-between gap-2 min-h-[30px]">
                  <div className="inline-flex items-center py-[6px] px-2.5 rounded-full bg-[#141922] border border-[#2A3342] min-w-0 sm:min-w-[96px] justify-center">
                    <span className="sm:text-sm text-xs font-semibold text-[#D0D6E1] whitespace-nowrap">
                      {total_sale + ` ${language["Sale"]}`}
                    </span>
                  </div>

                  {/* ✅ FIXED RATING */}
                  <div className="flex rtl:space-x-reverse space-x-1 items-center min-w-0 sm:min-w-[56px] justify-end">
                    <span className="text-[#FFD66B]">⭐</span>
                    <span className="sm:text-lg text-base font-medium leading-[21px] text-white">
                      {finalRating}
                    </span>
                  </div>
                </div>

                <div className="mt-2">
                  <div className="flex flex-col gap-1 min-h-[26px] sm:grid sm:grid-cols-[70px_1fr] sm:items-start sm:gap-2">
                    <span className="text-[10px] uppercase tracking-wide text-[#7C838D] whitespace-nowrap hidden sm:inline">
                      Review
                    </span>
                    <div className="h-[56px] w-full">
                      <ReviewBox reviews={finalReviewTexts} hasReviews={hasReviews} disableReview={disableReview} rotationEnabled={rotationEnabled} />
                    </div>
                  </div>
                  <div className="mt-2 h-[2px] w-full rounded-full bg-gradient-to-r from-[#F6C453] via-[#FFD36A] to-transparent opacity-80" />
                </div>
              </div>
            </div>
          </div>
        </div>
      </Link>
    );
  }

  /* ========================= COLUMN VIEW ========================= */
  return (
    <Link href={`/products/${slug}`} className="block">
      <div
        {...props}
        className={`product-col-item w-full group rounded-[22px] overflow-hidden relative ${
          props.className || "bg-[#0B0E14]"
        } border border-[#222734] shadow-[0_12px_30px_rgba(0,0,0,0.5)] transition-all duration-300 hover:-translate-y-1 hover:shadow-[0_26px_70px_rgba(0,0,0,0.65)] hover:border-[#364055]`}
      >
        <div className="pointer-events-none absolute inset-0 opacity-0 group-hover:opacity-100 transition-opacity duration-300 bg-[radial-gradient(120%_120%_at_0%_0%,rgba(80,140,255,0.18),transparent_60%)]" />
        <div className="absolute left-0 top-0 h-[3px] w-full bg-gradient-to-r from-[#6AE0FF] via-[#7C5CFF] to-[#00F0A8] opacity-70" />
        <div className="w-full p-3 sm:p-[12px] flex flex-col space-y-3 sm:space-y-4">
          <div className="w-full h-[170px] sm:h-[215px] relative rounded-xl sm:rounded-2xl overflow-hidden bg-gradient-to-br from-[#0F141B] to-[#0A0D12] flex items-center justify-center p-2 sm:p-3 ring-1 ring-[#1B2230] group-hover:ring-[#2B3550] transition-all duration-300">
            <img
              src={process.env.BASE_URL + thumbnail_image}
              alt=""
              className="w-full h-full object-contain transform scale-100 group-hover:scale-[1.05] transition-all duration-300 max-w-[92%] max-h-[92%] drop-shadow-[0_12px_30px_rgba(0,0,0,0.7)]"
            />
          </div>

          <div className="flex-1">
            <div className="w-full px-1 sm:px-2 pb-4 sm:pb-5 flex flex-col">
              {/* Price */}
              <div className="flex rtl:space-x-reverse space-x-2 items-center mb-1.5">
                <span className="text-lg sm:text-xl leading-6 font-bold text-[#7CFFB2] notranslate">
                  {offer_price
                    ? calculatePrice(offer_price)
                    : calculatePrice(regular_price)}
                </span>
                {offer_price && (
                  <span className="text-xs text-[#8B93A2] line-through notranslate">
                    {calculatePrice(regular_price)}
                  </span>
                )}
              </div>

              <h1 className="text-base sm:text-xl text-white leading-5 sm:leading-6 font-semibold line-clamp-3 mb-2 sm:mb-3 min-h-[54px] sm:min-h-[72px] notranslate">
                {name}
              </h1>

              <div className="flex items-center justify-between gap-2 min-h-[28px]">
                <div className="inline-flex items-center py-[5px] px-2 rounded-full bg-[#141922] border border-[#2A3342] min-w-0 sm:min-w-[96px] justify-center">
                  <span className="text-xs sm:text-sm font-semibold text-[#D0D6E1] whitespace-nowrap">
                    {total_sale + ` ${language["Sale"]}`}
                  </span>
                </div>

                {/* ✅ FIXED RATING */}
                <div className="flex rtl:space-x-reverse space-x-1 items-center min-w-0 sm:min-w-[56px] justify-end">
                  <span className="text-[#FFD66B]">⭐</span>
                  <span className="text-base sm:text-lg font-medium leading-[21px] text-white">
                    {finalRating}
                  </span>
                </div>
              </div>

              <div className="mt-2 sm:mt-2">
                <div className="hidden sm:grid grid-cols-[70px_1fr] items-start gap-2">
                  <span className="text-[10px] uppercase tracking-wide text-[#7C838D] whitespace-nowrap">
                    Review
                  </span>
                  <div className="h-[56px] w-full">
                    <ReviewBox reviews={finalReviewTexts} hasReviews={hasReviews} disableReview={disableReview} rotationEnabled={rotationEnabled} />
                  </div>
                </div>
                <div className="sm:hidden h-[56px] w-full">
                  <ReviewBox reviews={finalReviewTexts} hasReviews={hasReviews} disableReview={disableReview} rotationEnabled={rotationEnabled} />
                </div>
                <div className="mt-2 h-[2px] w-full rounded-full bg-gradient-to-r from-[#F6C453] via-[#FFD36A] to-transparent opacity-80" />
              </div>
            </div>
          </div>
        </div>
      </div>
    </Link>
  );
}

export default ProductCard;
