"use client";

import {
  useGetMessageQuery,
  useSendMessageMutation,
} from "@/store/features/message/apiSlice";
import React, { useEffect, useRef, useState } from "react";
import ReactTimeAgo from "react-time-ago";
import StringLang from "@/utilities/StringLang";
import { useSearchParams } from "next/navigation";

function OrderChatBoard({ id: propId, isApprove }) {
  const searchParams = useSearchParams();
  const orderId = searchParams.get("orderId") || propId;

  const [message, setMessage] = useState(""); // Fixed: Initialize as empty string
  const [image, setImage] = useState(null);

  const { data, isFetching } = useGetMessageQuery(
    { id: orderId },
    { skip: !orderId }
  );

  const [sendMessage, { isLoading }] = useSendMessageMutation();

  const bottomRef = useRef(null);

  /* ================= AUTO SCROLL ================= */
  useEffect(() => {
    if (bottomRef.current) {
      bottomRef.current.scrollIntoView({ behavior: "smooth" });
    }
  }, [data?.messages]);

  /* ================= ROLE UI ================= */
  const roleUI = (msg) => {
    if (Number(msg.send_seller) === 1) {
      return {
        label: "Author",
        align: "justify-start",
        bubble: "bg-black text-white",
      };
    } else if (Number(msg.send_seller) === 2) {
      return {
        label: "ADMIN",
        align: "justify-start",
        bubble: "bg-blue-600 text-white",
      };
    } else {
      return {
        label: "YOU",
        align: "justify-end",
        bubble: "bg-green-600 text-white",
      };
    }
  };

  /* ================= SEND MESSAGE ================= */
  const handleSend = async () => {
    if (!message && !image) return;

    const formData = new FormData();

    if (message?.trim()) {
      formData.append("message", message.trim());
    }

    if (image) {
      formData.append("attachment", image);
    }

    try {
      await sendMessage({
        formData,
        id: orderId,
        resetForm: () => {
          setMessage("");
          setImage(null);
        },
      }).unwrap();
    } catch (err) {
      console.error("Send failed", err);
    }
  };

  return (
    <div className="w-full h-[75vh] flex flex-col bg-black border border-[#23262B] rounded-lg overflow-hidden">
      {/* ================= CHAT LIST ================= */}
      <div className="flex-1 overflow-y-auto px-4 py-4 space-y-4">
        {data?.messages?.map((msg) => {
          const ui = roleUI(msg);

          return (
            <div key={msg.id} className={`flex ${ui.align}`}>
              <div className="flex flex-col max-w-[75%]">
                {/* META */}
                <span className="text-[11px] text-gray-400 mb-1">
                  {ui.label} • <ReactTimeAgo date={new Date(msg.created_at)} />
                </span>

                {/* MESSAGE */}
                <div
                  className={`px-4 py-3 rounded-2xl break-words ${ui.bubble}`}
                >
                  {/* IMAGE */}
                  {msg.attachment && (
                    <img
                      src={`${process.env.NEXT_PUBLIC_ADMIN_URL}/public/uploads/messages/${msg.attachment}`}
                      alt="chat"
                      className="max-w-[220px] rounded-lg mb-2"
                    />
                  )}

                  {/* TEXT */}
                  {msg.message && msg.message !== "Image Attached" && (
                    <div
                      style={{ whiteSpace: "pre-wrap" }}
                      dangerouslySetInnerHTML={{ __html: msg.message }}
                    />
                  )}
                </div>
              </div>
            </div>
          );
        })}

        <div ref={bottomRef} />
      </div>

      {/* ================= INPUT ================= */}

      <div className="border-t border-[#23262B] p-3 bg-black">
        <textarea
          value={message}
          onChange={(e) => setMessage(e.target.value)}
          placeholder="Type your message…"
          className="
              w-full
              min-h-[70px]
              max-h-[130px]
              resize-none
              rounded-xl
              bg-[#0B0E12]
              border border-primary-blue
              p-3
              text-white
              focus:outline-none
              mb-2
            "
        />

        <input
          type="file"
          accept="image/*"
          onChange={(e) => setImage(e.target.files?.[0] || null)}
          className="text-white text-sm mb-2"
        />

        <button
          onClick={handleSend}
          disabled={isLoading}
          className="
              w-full
              py-3
              rounded-xl
              bg-primary-blue
              text-black
              font-semibold
              disabled:opacity-60
            "
        >
          <StringLang string={isLoading ? "Sending…" : "Send Message"} />
        </button>
      </div>
    </div>
  );
}

export default OrderChatBoard;
