"use client";

import React from "react";
import ProductCard from "@/components/Common/Card/ProductCard";
import { useSellerProductsQuery } from "@/store/features/seller/apiSlice";
import SvgLoader from "@/components/Helper/Loader/SvgLoader";
import PaginateObserver from "@/components/Helper/PaginateObserver";
import { useDispatch, useSelector } from "react-redux";
import { sellerProductPageIncrease } from "@/store/features/pagination/paginationSlice";
import StringLang from "@/utilities/StringLang";

function SellerProfile({ datas }) {
  const dispatch = useDispatch();
  const page = useSelector((state) => state.pagination.sellerProductPage);

  const paginationHandler = () => {
    dispatch(sellerProductPageIncrease());
  };

  const { data, isFetching } = useSellerProductsQuery({
    page,
    userName: datas?.author?.user_name,
  });

  if (!datas || !datas.author) return null;

  const { author, total_sale, average_rating } = datas;

  // ADMIN MANUAL VALUES
  const manualTotalSales = author.manual_total_sales;
  const manualRating = author.manual_rating;
  const manualDeliveryTime = author.manual_delivery_time;

  // FINAL RATING SOURCE
  const finalRating =
    manualRating !== null && manualRating !== undefined
      ? Number(manualRating)
      : Number(average_rating ?? 0);

  // ⭐ STAR COMPONENT
  const StarRating = ({ rating }) => (
    <div className="flex items-center space-x-1">
      {[1, 2, 3, 4, 5].map((i) => (
        <svg
          key={i}
          width="18"
          height="18"
          viewBox="0 0 24 24"
          fill={i <= Math.round(rating) ? "#FACC15" : "none"}
          stroke="#FACC15"
          strokeWidth="1.5"
        >
          <path d="M12 2l3.09 6.26L22 9.27l-5 4.87L18.18 22 12 18.56 5.82 22 7 14.14 2 9.27l6.91-1.01L12 2z" />
        </svg>
      ))}
    </div>
  );

  return (
    <div className="w-full mt-11 pb-[100px]">
      <div className="theme-container mx-auto">
        <div className="w-full lg:px-[190px]">

          {/* SELLER HEADER */}
          <div className="w-full bg-black px-[29px] py-10 rounded-lg mb-[30px]">
            <div className="flex space-x-4 items-center">
              <div className="w-[100px] h-[100px] rounded-full overflow-hidden">
                <img
                  src={process.env.BASE_URL + author.image}
                  className="w-full h-full object-cover"
                />
              </div>

              <div>
                <h1 className="text-white text-xl font-bold">
                  {author.name}
                </h1>

                {/* ⭐ FULL STAR DISPLAY */}
                <div className="flex items-center space-x-2 mt-2">
                  <StarRating rating={finalRating || 5} />
                  <span className="text-white font-semibold text-lg">
                    {finalRating ? finalRating.toFixed(1) : "5.0"}
                  </span>
                </div>
              </div>
            </div>

            {/* STATS */}
            <div className="grid sm:grid-cols-3 gap-5 mt-8">

              {/* TOTAL SALES */}
              <div className="bg-[#0B0E13] px-5 py-6 rounded-md">
                <p className="text-lg">
                  <StringLang string="Total Sales" />
                </p>
                <p className="text-2xl text-white font-bold">
                  {manualTotalSales ?? total_sale ?? 0}
                </p>
              </div>

              {/* OVERALL RATING */}
              <div className="bg-[#0B0E13] px-5 py-6 rounded-md">
                <p className="text-lg">
                  <StringLang string="Overall ratings" />
                </p>
                <div className="flex items-center space-x-2 mt-1">
                  <StarRating rating={finalRating || 5} />
                  <span className="text-white font-bold text-xl">
                    {finalRating ? finalRating.toFixed(1) : "5.0"}
                  </span>
                </div>
              </div>

              {/* DELIVERY */}
              <div className="bg-[#0B0E13] px-5 py-6 rounded-md">
                <p className="text-lg">
                  <StringLang string="Time" />
                </p>
                <p className="text-2xl text-white font-bold">
                  {manualDeliveryTime ?? "Instant"}
                </p>
              </div>
            </div>

            {/* ABOUT */}
            <h2 className="text-2xl font-bold text-white mt-10 mb-3">
              <StringLang string="About Seller" />
            </h2>

            <p
              className="text-base"
              dangerouslySetInnerHTML={{
                __html: author?.about_me?.replace(/<[^>]*>/g, ""),
              }}
            />
          </div>

          {/* PRODUCTS */}
          <p className="text-2xl font-bold text-white mb-4">
            <StringLang string="Seller Products" />
          </p>

          {data?.products?.data?.length ? (
            <PaginateObserver
              currentPage={page}
              lastPage={data.products.last_page}
              loading={isFetching}
              handler={paginationHandler}
              type="onclick"
            >
              <div className="grid xl:grid-cols-3 sm:grid-cols-2 gap-6">
                {data.products.data.map((p, i) => (
                  <ProductCard {...p} key={i} />
                ))}
              </div>
            </PaginateObserver>
          ) : isFetching ? (
            <SvgLoader />
          ) : (
            <p className="text-center text-sm">
              <StringLang string="No Products Found!" />
            </p>
          )}
        </div>
      </div>
    </div>
  );
}

export default SellerProfile;